% Price a payoff given a trajectory set
% 
% [V]=valueindex(A,I,s0,K,delta,payoff)
% 
% Inputs:
% 
% A and I (from creatematrix.m )
% s0 - the initial stock value in dollars
% K - the strike
% \delta - see [2]
% 
% Outputs:
% 
% V a  1x(5-6)x(number of possible nodes) array consisting of 5 or 6 dimensional
% tuples where the first 2 (in type I models) or 3 (in type II models)
% entries are the same in the index I followed by the stock value, the
% hedging value (value of bank account and stock holdings) and the number
% of the underlying shares.
% 
% [v,h]=convexhull(n,s0) calculates the hedging value and the number of
% shares calculated by the convex hull algorithm. See the explanations in
% [1]
% 
% Inputs
% n: array of previous stock and option values
% s0: current stock value
% 
% 
% References 
% [1] Algorithm XXX: Trajectorial Asset Models in Matlab
% [2] Trajectorial Asset Models with Operational Assumptions
% Both papers by S.Ferrando and A. Fleck.
function [V]=valueindex(A,I,s0,K,delta,payoff)

Strikes=length(K);

[r1,r2,r3]=size(I);

V=zeros(r1,r2+1+2*Strikes,r3); %for S, z's and h's
   
V(1,1:r2,:)=I; 

%find nodes corresponding to time T

zrange=find(I(1,r2,:)==max(I(1,r2,:)));

%fill with option payoffs

for i=transpose(zrange)

    if sum(A(:,i))>0

    k=I(1,1,i);
    Sk=s0+k*delta;
    V(1,r2+1,i)=Sk;
    
    for st1=1:Strikes

        z=feval(payoff,Sk,K(st1));
        V(1,r2+2*st1,i)=z;

        %In order to make sure we can't have negative stock values. Represents
        %a bankrupt or de-listed company.

        if Sk<=0  
             V(1,r2+1,i)=-Inf;
             V(1,r2+2*st1,i)=feval(payoff,0,K(st1)); %just the intrinsic value   
        end

    end
    
    end
end

%price the rest

l=length(I);
last=length(zrange);

%look at nodes working backwards

Atrans=A';

 for r=l-last:-1:1 %bottom rows
     
     col=find( Atrans(:,r) );
     
     if isempty(col)~=1 %make sure node is connected   
          
         k=I(1,1,r);
         Si=s0+k*delta;
         V(1,r2+1,r)=Si;

         for st2=1:Strikes
         
             %here too- once you go bankrupt or get delisted, that's it.

             if Si<=0 
                 V(1,r2+1,r)=-Inf;
                 V(1,r2+2*st2,r)=feval(payoff,0,K(st2)); %just the intrinsic value   
             end

             n=[];
             
             for kk=1:length(col) %over the different columns    

                     vS=V(1,r2+2*st2,col(kk));
                     S =V(1,r2+1,col(kk));
                     n=[n;vS,S];
                                     
             end   

             if size(n)==[1,2]    
                V(1,r2+2*st2,r)=n(1); %special case, no change in value.
             else   

             [v,h]=convexhull(n,Si);

             V(1,r2+2*st2,r)=v;
             V(1,r2+1+2*st2,r)=h;

             end 
         
         end
     
     end
 end

end

function [v,h]=convexhull(n,s0)

l=length(n);

v=0;
h=0;

ymax=-Inf; 

for i=1:l
    for j=1:l
        
       if n(i,2)~=n(j,2) && n(i,2)>=s0 && n(j,2)<s0 %see section 2 and 5.1 in [?]
        
       x=( n(i,1)-n(j,1) )/( n(i,2)-n(j,2) );
       y=( n(i,1) - x*(n(i,2)-s0) );
       
           if y>=ymax
              ymax=y; 
              v=ymax;
              h=x;
           end
       
       end
        
    end
end


% %As mentioned in [?], can reformulate as a linear programming problem in
% %the following way:
% 
% l=length(n);
% 
% b=n(:,1);
% A=[ones(l,1),n(:,2)-s0*ones(l,1)];
% f=[1;0];
% 
% %using standard MATLAB linear programming 
% %x = linprog(f, A, b,Aeq,beq ,lb       ,ub)
% %x = linprog(f,-A,-b,[] ,[]  ,[-1;-0.5],[Inf;Inf]);
% x = linprog(f,-A,-b);
% 
% v=x(1);
% h=x(2);

end

