
% Model Parameter Estimation
% 
%     [Gamma_array,tau_array,N_array,n1,n0,V]=parameter_est(M_T, data, delta_0 , delta ) 
%     
%     Inputs:
% 
%     A set ofstock price data (data) 
%     The time till expiry of an option written on said stock (M_T) 
%     (in units of \Delta, and the refinements \delta_0 and
%     \delta outlined in [1] and [2]) 
% 
%     Outputs:
%     Arrays (Gamma_array,tau_array,N_array) that give the parameter estimates for
%     \Gamma, \tau and number of rebalances N in every sub-interval (window) in
%     the data of length M_T. 
% 
%     Additionally it outputs n1,n0 and V. 
% 
%     n1 represents the empirical conditional trajectory N_A given in section 7 in
%     [2]; that is, the conditional stock movements greater than \delta. n0
%     represents the remaining conditional stock movements, that is those less
%     than \delta. Finally, V gives the variation data for each sub-interval of
%     length M_T.
% 
% References 
% [1] Algorithm XXX: Trajectorial Asset Models in Matlab
% [2] Trajectorial Asset Models with Operational Assumptions
% Both papers by S.Ferrando and A. Fleck.

function [Gamma_array,tau_array,N_array,n1,n0,V]=parameter_est(M_T, data, delta_0 , delta )

l=length(data); 
n=l-M_T;

%time parameter array
N_array=zeros(1,n+1);
Gamma_array=zeros(1,n+1);
tau_array=zeros(1,n+1);

%V_\rho for each interval 

V=zeros(n+1,M_T+1);

n1=[];
n0=[];

%split data into subintervals of length T for estimation
for j=0:n
    
    a=j+1; %begining of interval
    b=M_T+j; %end of interval
    interval=data(a:b);  
    
    %want to look at \delta_0 and \delta moves 
    
    r_array=[1]; 
    t_array=[1];
    
    k=1;
    while k<M_T-1     
        for kk=k:M_T
            if abs(interval(k)-interval(kk))>=delta_0    
                r_array=[r_array;kk];              
                break            
            end          
        end      
        k=kk;     
    end
    
    k=1;
    while k<M_T-1 
        for kk=(r_array(r_array>k))' %guarntee t's are in our sampling times
            if abs(interval(k)-interval(kk))>=delta
                t_array=[t_array;kk];
                break
            end
        end
        k=kk;   
    end

    %now that we have the out sampling and rebalanicng times we can get our
    %parameter estimates for this interval
    
    %time parameters 
    if ~isempty(t_array(2:length(t_array)))
    
        N_array(j+1)=length(t_array)-1;
        tau_array(j+1)=t_array(length(t_array)); %not minus the first time as it will be zero in this interval
        Gamma_array(j+1)=min(diff(t_array)); %minimum wait for a delta increment
    
    else 
        
        N_array(j+1)=0;
        tau_array(j+1)=0;
        Gamma_array(j+1)=M_T;
        
    end
    
    
    %partial estimates for N^{E} which I call n1 in the code
    %also need a n0 for less than delta moves 
    
    %want to interploate chart:
    %need to make sure interpolation does not throw away data- need to make
    %sure range of interpolation function is large enough: hence the
    %max and min
    
    D=diff(data);
    
    round_target_min=(min(D)-mod(min(D),delta_0)-delta_0);
    round_target_max=(max(D)-mod(max(D),delta_0)+delta_0);
    rounded_range=round_target_min:delta_0:round_target_max;

    %interpolate chart by approximating with the nearest point available
    box_x=interp1(rounded_range,rounded_range,diff(interval(t_array)),'nearest');
    
    %corresponds to eq 8 in summary 
    n1_array=round([ (box_x)/delta_0 , diff(t_array) ]);
    
    %rounding handles prescison issues-guearntees we get integers
    
    n1_array=unique(n1_array,'rows','stable');
    n1=[n1;n1_array];
    n1=unique(n1,'rows','stable');
    
    %now want \delta_0 moves after the last \delta move -> use r's i.e
    %sampling time
    
    indx=find(r_array==tau_array(j+1));    
    last_times=r_array(indx:length(r_array));
    
    %box_x_0=interp1(rounded_range,rounded_range,diff(interval([last_times;M_T])),'nearest');
    box_x_0=interp1(rounded_range,rounded_range,diff(interval(last_times)),'nearest');
    
    %n0_array=round( [ (box_x_0)/delta_0 , diff([last_times;M_T]) ] );
    n0_array=round( [ (box_x_0)/delta_0 , diff(last_times) ] );
    n0_array=unique(n0_array,'rows','stable');
    n0=[n0;n0_array];
    n0=unique(n0,'rows','stable');
    
    %want V_\rho  estimate now 
    
    %changes= [ r_array' ; 0, abs(  interp1(rounded_range,rounded_range,diff(interval(r_array)),'nearest') )' ];
    changes= [ r_array' ; 0, abs(  diff(interval(r_array)) )' ];
    
    new_range=0:delta_0:10*M_T*max(abs(n1(:,1)))*delta_0;
    
    for p=1:M_T+1 %arrays start at 1 not zero
    
    %V(j+1,p)= sum(changes(2,r_array<=p)); 
    V(j+1,p)= interp1(new_range,new_range,sum(changes(2,r_array<=p)),'nearest')/delta_0;
    
    end
    
end

end 