% Construct the trajectory set
% 
% [As,I] = creatematrix( N0, N, model, Qup, Qdown )
%
% [I]=index(N)
% 
% Inputs:
% N- gives the maximal ranges for the (k,j,n) triples outlines in
% section 4 of [2]
% N0- gives the (k,j,n) triple defining the initial node.
% “model” is the name of the model file in question being used- see section
% 2.4.
% Qup and Qdown are 1xM_T arrays that give the maximum and
% minimum value of the admissible variation needed for type II models. 
% 
% Outputs:
% 
% A- a sparse matrix representing the adjacency matrix
% I- the index as described in [1].
% 
% 
% 
% 
% References 
% [1] Algorithm XXX: Trajectorial Asset Models in Matlab
% [2] Trajectorial Asset Models with Operational Assumptions
% Both papers by S.Ferrando and A. Fleck.

function [As,I] = creatematrix( N0, N, model, Qup, Qdown )

I=index(N); %makes integer grid index
Nmax=length(I); %all the possible nodes in the Adjacency matrix 
Dim=length(N);
Dim_plus_one=Dim+1;

Kmax=N(1);
Jmax=N(2);

child_array=index([N,0]);
parent_array=index([N,0]);

CHILDREN=[];
PARENT=[];

%find the node corresponding to the initial stock price
for i=1:Nmax 
    if I(1,:,i)==N0  
    o=i;
    break
    end
end

%set it in the parent array
parent_array(1,Dim_plus_one,o)=1;

while any(parent_array(1,Dim_plus_one,:)==1)

%make sure the child array is clear    
child_array(1,Dim_plus_one,child_array(1,Dim_plus_one,:)==1)=0;    
    
%find parents
for i=transpose(find(parent_array(1,Dim_plus_one,:)==1))

    %make sure it has not already produced children
    
    if ~any(bsxfun(@eq,i,PARENT))
        
        %pick up parent find its children
        n=parent_array(1,1:Dim_plus_one-1,i);
       
        New_children=feval(model,n,Qup,Qdown);
       
        %add info to matrix
            
        if size(New_children)==[1,length(N)]
            num_of_child=1;
        else 
            [num_of_child,~]=size(New_children) ;
        end
        
        fnew=[];
        
        for ii=1:num_of_child
            
           nprime=New_children(ii,:);
           
           if Dim==2
               k=nprime(1);
               j=nprime(2);

               f=j*(2*Kmax +1) + (k + Kmax+1);

           end
           
           if Dim==3
               k=nprime(1);
               j=nprime(2);
               n=nprime(3);
               
               f=j*(2*Kmax +1) + (k + Kmax+1) + n*( Jmax*(2*Kmax +1) + (2*Kmax+1) );
           end
            
           if f<=Nmax   %make sure N is in the grid
                  fnew=[fnew;f];
                  
                  child_array(1,Dim_plus_one,f)=1;
           end
             
        end
        
        newlength=length(fnew);
        PARENT=[PARENT;i*ones(newlength,1)];
        CHILDREN=[CHILDREN;fnew];
               
    end
    
end


parent_array=child_array;

end

PARENT=[PARENT;Nmax];
CHILDREN=[CHILDREN;Nmax];

ASlength=length(PARENT);
As=sparse(PARENT,CHILDREN,[ones(ASlength-1,1);0]);

end

function [I]=index(N)

l=length(N);

Kmax=N(1);
Jmax=N(2);

L=(2*Kmax+1)*(Jmax+1);

LENGTH=L;

for i=3:l
    LENGTH=LENGTH*(N(i)+1);  
end

I=zeros(1,l,LENGTH);

n=1;

%fills first "frame" that is (k,j) as k is unique-can be negative 

for j=0:Jmax
    for i=-Kmax:1:Kmax
        
        I(1,1,n)=i;
        I(1,2,n)=j;
        n=n+1;
        
    end
end

if l>2
    for i=3:l
        
        A=I(1,1:(i-1),1:L);
        
        Ap=repmat(A,1,1,N(i));
        
        I(1,1:(i-1),(L+1):(L*(N(i)+1)))=Ap;
        
        for j=1:N(i)
            
            I(1,i,(j*L+1):((j+1)*L))=j;    

        end
        
        L=L*(N(i)+1);
        
    end
end

end

