#!/usr/bin/python3
# -----------------------------------------------------------------
# 03 - runSobolOnWholeModel.py
#
# Authors: B.Tarraf, M.Leguebe
#
# Performs a global sensitivity analysis using Sobol indices
# (algorithms from SALib library)
#
# The number of samples should be very large, you should replace
# the call to 'reduce' with a 'reduceOnSlurmCluster' if you have
# access to such a machine. (then you will need to edit the file
# reduceOnCluster.py in pyCompMito)
# Here the number of samples is kept low to make sure it runs on
# a laptop with several cores.
#
# -----------------------------------------------------------------

import time
import numpy        as np
from SALib.sample   import saltelli
from SALib.analyze  import sobol
from scipy.optimize import curve_fit

# Uncomment the following line and set the path 
# or use PYTHONPATH environment variable :
#sys.path.insert(0,'/path/to/pyCompMito/')
from pyCompMito import *

# Experiment details and data
# (see pyCompMito/experiments.py for a list of experiments)
manip = manips["data16h00"]

# Location where temporary file will be written 
# (anticipate large amount of data => not in home)
tmpDir = './results'

# Number of Sobol samples (should be >1000 for efficient measure)
nSamples = 2
# The problem described with the SALib format 
# (dictionnary with 'num_vars','bounds','names', 
#  see pyCompMito/mitoModels/TCLD2019.py)
SALibPb  = SALibTcld7Params

# The model Y(P)
model      = runSingleTCLD
# The distance function D(Y(P))
functional = costFunc

# Number of cpus on which evaluations of f will be performed
nCpus      = 4
# Estimated duration in HOURS of a single evaluation of f on a single CPU
timePerRun = 0.07

# =================================================================

print("Computing samples")
X = saltelli.sample(SALibPb,nSamples,calc_second_order=False)

print("Evaluating model for {:d} sets of parameters. Be patient.".format(X.shape[0]))

# Here we launch X.shape[0] computations of 'functional' on the cluster, 
# using nCpus cpus, each evaluation of 'functional' being estimated to last 0.01hour.

# The second argument (X.T,paramFile,np.arange(X.shape[0])) is the tuple of arguments given to 'function'.
# Numpy arrays with last dimension being X.shape[0] will be distributed between processes,
# such that each CPU does its part of the work.

# See reduce and reduceOnSlurmCluster in paralleltools.py 
# for better explanation on arguments (and also optional arguments)

#Y = reduceOnSlurmCluster(functional,(model,X.T,paramFile,'data16h00'),X.shape[0],nCpus,timePerRun,tmpDir,deleteFiles=False)

# Laptop version:
Y = reduce(functional,(X.T,model,manip),X.shape[0],nCpus,progressBar=True)

Xfail = X[Y<0,:]
print("{:d}/{:d} parameter sets led to failed simulations".format(Xfail.shape[0],X.shape[0]))
#print(Xfail)

# The number of evaluations must be a power of 2
# so we replace the missed evaluations by the mean
# of all other
Y[Y<0]=np.mean(Y[Y>=0.])

# Get the results of the analysis
Si = sobol.analyze(SALibPb,Y,calc_second_order=False,print_to_console=False)

# ==================================================
# Plot the results

import matplotlib.pyplot as plt

fs = 22
plt.ion()
fig, ax = plt.subplots()
ax.bar            (SALibPb['names'], Si['ST'],color='peru',label=r'$S_T$')
#ax.bar            (SALibPb['names'], Si['S1'],color='peru',label=r'$S_1$')
ax.set_ylabel     (r'Total Sobol indices',fontsize=fs+2)
#ax.set_ylabel     (r'First order Sobol indices',fontsize=fs+2)
ax.set_yticklabels(range(0,0.8,0.1),rotation=0,fontsize=fs+2)
ax.set_xlabel     (r'Parameters',fontsize=fs+2)
ax.set_xticklabels(SALibPb['names'],rotation=0,fontsize=fs+2)
ax.legend         (fontsize=fs)
plt.show()


