#!/usr/bin/python3
# -----------------------------------------------------------------
# 02 - runSingleTCLD.py
#
# Authors: B.Tarraf, M.Leguebe
#
# Script for running a single computations of our mitochondrial
# model.
# Usage:
# ./runSingleTCLD.py paramFile1 paramFile2 ...
#
# where parameter files are located in the parameters directory
# -----------------------------------------------------------------

import os
import sys
import copy
import shutil
import numpy as np
import time

# Uncomment the following line and set the path 
# or use PYTHONPATH environment variable :
#sys.path.insert(0,'/path/to/pyCompMito/')
from pyCompMito import *

# =================================================================
# Parse command line: get configuration files, parse options

paramFiles = []
optHelp    = False
optClean   = False
optPlot    = False
for arg in sys.argv[1:]:
  if (getExtension(arg) == "init"):
    paramFiles.append(arg)
  elif (arg == "-h"  or arg == "--help"):
    optHelp  = True
  elif (arg == "-c"  or arg == "--clean"):
    optClean = True
  elif (arg == "-p"  or arg == "--plot"):
    optPlot  = True

if (optHelp or len(paramFiles)==0):
  print("Usage:")
  print("  " +sys.argv[0]+" paramsFile1.init paramsFile2.init ... ")
  print("Options:")
  print(" -h  : print help (--help)")
  print(" -c  : clean existing files in output directory (--clean)")
  print(" -p  : plot results and save .pngs (--plot)")
  print(" -ns : do not solve ODEs, plot only (--nosim)")
  sys.exit()

# =================================================================
# Run simulations

for paramFile in paramFiles:

  # Create configuration, output directory, 
  # and save copy of paramFile.
  config = myConfigParser(paramFile)
  outDir = config('OutDir','./')

  if optClean:
    print("Cleaning files in " + outDir)
    shutil.rmtree(outDir,ignore_errors=True)

  mkdir_p(outDir)
  try:
    shutil.copy(paramFile,outDir)
  except:
    print("Error while copying parameter file. File already in output directory and so not overwritten (?)")

  # Create solver
  solver = getODESolver(config)
  model  = TCLD_2019_all_param(config)
  Y0 = model.initializeStateVar(solver)
  t=time.time()
  solver.run(model,Y0,outDir+'/res.dat')

  if optPlot:
    from matplotlib.pyplot import *
    ion()
    # Set font
    rc('text',usetex=True)
    rc('font',family='serif')
    fs = 18

    Ydata  = np.loadtxt(outDir+'/res.dat',comments='#')

    # Model variables
    fig,axes = subplots(3,2,figsize=(12,8))
    axes[0][0].plot(Ydata[:,0],Ydata[:,1],linewidth=2)
    axes[0][0].set_ylabel(r'$\Delta E_\mathrm{resp}$',fontsize=fs)
    axes[0][1].plot(Ydata [:,0],Ydata [:,2],linewidth=2)
    axes[0][1].set_ylabel(r'$\Delta p$',fontsize=fs)
    axes[1][0].plot(Ydata [:,0],model.DGpm(Ydata[:,3]),linewidth=2)
    axes[1][0].set_ylabel(r'$\Delta G_\mathrm{pm}$',fontsize=fs)
    axes[1][1].plot(Ydata [:,0],model.DGpc(Ydata[:,4]),linewidth=2)
    axes[1][1].set_ylabel(r'$\Delta G_\mathrm{pc}$',fontsize=fs)
    axes[2][0].plot(Ydata [:,0],Ydata [:,5],linewidth=2)
    axes[2][0].set_ylabel(r'Ca$_\mathrm{m}$',fontsize=fs)
    axes[2][1].plot(Ydata [:,0],Ydata [:,6],linewidth=2)
    axes[2][1].set_ylabel(r'Ca$_\mathrm{c}$',fontsize=fs)
    fig.tight_layout()

    # Fluxes
    fig,axes = subplots(3,2,figsize=(12,8))
    axes[0][0].plot(Ydata[:,0],model.jHrespFit((Ydata[:,1],Ydata[:,2]),model.p0,model.p1,model.p2,model.p3,model.p4,model.p5),linewidth=2)
    axes[0][0].set_ylabel(r'$J_\mathrm{Hresp}$',fontsize=fs)

    axes[0][1].plot(Ydata [:,0],model.jHF1F0Fit((Ydata [:,3],Ydata[:,2]),model.p10,model.p11,model.p12,model.p13,model.p14,model.p15),linewidth=2)
    axes[0][1].set_ylabel(r'$J_\mathrm{HF1F0}$',fontsize=fs)

    axes[1][0].plot(Ydata[:,0],model.jANT((Ydata[:,3],Ydata[:,4],Ydata[:,2]),model.p16,model.p17,model.p18,model.p19),linewidth=2)
    axes[1][0].set_ylabel(r'$J_\mathrm{ANT}$',fontsize=fs)

    axes[1][1].plot(Ydata [:,0],model.jHUniFit((Ydata [:,6],Ydata[:,2]),model.p24,model.p25,model.p26),linewidth=2)
    axes[1][1].set_ylabel(r'$J_\mathrm{uni}$',fontsize=fs)

    axes[2][0].plot(Ydata [:,0],model.JNaCa(Ydata [:,5],model.p22,model.p23),linewidth=2)
    axes[2][0].set_ylabel(r'$J_\mathrm{NaCa}$',fontsize=fs)

    axes[2][1].plot(Ydata [:,0],model.jleak(Ydata [:,2],model.p20,model.p21),linewidth=2)
    axes[2][1].set_ylabel(r'$J_\mathrm{Hleak}$',fontsize=fs)
    fig.tight_layout()
