# -----------------------------------------------------------------
# MkFluxes.py
#
# Authors: B.Tarraf, M.Leguebe
# 
# Fluxes from the litterature (Bertram, Magnus & Keizer) 
# or derived from Altman-King-Hill diagrams as MK did
#
# See section 2.1 and appendix B of the paper for derivation
# of the MK fluxes
#
# -----------------------------------------------------------------
import numpy as np
from .commonParams import *

aTot = 12
kF1  = 1.71e6 # mM
piM  = 20     # mM
piC  = 16

deltaPh   = -0.4
deltaPsiB = 50   # in mV
rhoResp   = 0.4  # MK97 value in nmol/mg protein

def jHresMk(deltaEResp,deltaPsi):

    ''' Rates of the AKH diagram for the respiratory complex,
        some are constant, others are functions of deltaEresp
        and deltaPsi
    '''
    a12 = 4.e2
    a21 = 5
    a32 = 8.e4
    a34 = 4.e2
    a45 = 40
    a54 = 0.4
    a56 = 1.e2
    a65 = 1.e5
    a25 = 0 # No slip no leaks
    a52 = 0 # No slip no leaks

    eResp  = np.exp(FRT*deltaEResp)
    eRespK = np.exp(FRT*deltaEResp-lKResp)
    nadhM  = nadTot*eRespK**2/(1+eRespK**2)
    nadM   = nadTot-nadhM

    a23Star = 3.405e4
    a23     = a23Star*np.sqrt(nadhM)
    #a23     = 1.88*10**4

    a43Star = 159
    a43     = a43Star*np.sqrt(nadM)
    #a43     = 441

    ePsi  = np.exp(3*FRT*deltaPsi)
    ePsiB = ePsi/np.exp(3*FRT*deltaPsiB)

    a160 = 1.3e2
    a16  = a160*ePsiB

    a610 = 1.e12
    a61  = a610/ePsiB

    # Rates are all defined, now compute the coefficients
    # of the diagram

    # Pi A
    piAM = a16*a65*a54*a43*a32*a21
    piAP = a61*a56*a45*a34*a23*a12

    # Pi B
    piBM = a16*a65*a52*a21
    piBP = a61*a56*a25*a12

    # Pi C
    piCM = a25*a54*a43*a32
    piCP = a52*a23*a34*a45

    sigmaB = a43*a32+a34*a45+a32*a45
    sigmaC = a61*a12+a16*a65+a12*a65

    # S1 = sum of S1i
    S11 = a61*a56*a45*a34*a23
    S12 = a65*a54*a43*a32*a21
    S13 = a21*a61*a56*a45*a34
    S14 = a61*a21*a32*a43*a54
    S15 = a56*a61*a43*a32*a21
    S16 = a45*a56*a61*a32*a21
    S17 = a61*a56*a25*a34*a45
    S18 = a61*a56*a25*a32*a45
    S19 = a61*a56*a25*a32*a43
    S110 = a65*a21*a52*a32*a43
    S111 = a21*a65*a52*a32*a45
    S112 = a21*a65*a52*a34*a45
    S113 = a61*a21*a52*a32*a43
    S114 = a61*a21*a52*a32*a45
    S115 = a61*a21*a52*a34*a45
    S1 = S11+S12+S13+S14+S15+S16+S17+S18+S19+S110+S111+S112+S113+S114+S115

    # S2 = sum of S2i
    S21 = a34*a45*a56*a61*a12
    S22 = a16*a65*a54*a43*a32
    S23 = a45*a56*a61*a12*a32
    S24 = a12*a65*a54*a43*a32
    S25 = a43*a32*a56*a61*a12
    S26 = a61*a12*a54*a43*a32
    S27 = a61*a12*a52*a43*a32
    S28 = a16*a65*a52*a32*a43
    S29 = a61*a12*a52*a34*a45
    S210 = a12*a65*a52*a34*a45
    S211 = a16*a65*a52*a32*a45
    S212 = a16*a65*a52*a34*a45
    S213 = a12*a65*a52*a32*a43
    S214 = a12*a65*a52*a32*a45
    S215 = a61*a12*a52*a32*a45
    S2 = S21+S22+S23+S24+S25+S26+S27+S28+S29+S210+S211+S212+S213+S214+S215

    # S3 = sum of S3i
    S31 = a61*a56*a45*a23*a12
    S32 = a16*a65*a54*a43*a21
    S33 = a16*a65*a54*a43*a23
    S34 = a61*a56*a43*a23*a12
    S35 = a61*a54*a43*a23*a12
    S36 = a65*a54*a43*a23*a12
    S37 = a61*a12*a23*a52*a45
    S38 = a16*a65*a52*a23*a45
    S39 = a12*a65*a52*a23*a45
    S310 = a16*a65*a25*a43*a45
    S311 = a61*a12*a25*a43*a54
    S312 = a12*a65*a25*a43*a54
    S313 = a61*a12*a52*a23*a43
    S314 = a16*a65*a52*a23*a43
    S315 = a12*a65*a52*a23*a43
    S3 = S31+S32+S33+S34+S35+S36+S37+S38+S39+S310+S311+S312+S313+S314+S315

    # S4 = sum of S4i
    S41 = a61*a56*a34*a23*a12
    S42 = a16*a65*a54*a32*a21
    S43 = a16*a65*a54*a34*a21
    S44 = a61*a54*a34*a23*a12
    S45 = a16*a65*a54*a34*a23
    S46 = a65*a54*a34*a23*a12
    S47 = a61*a12*a52*a23*a34
    S48 = a12*a65*a52*a23*a34
    S49 = a16*a65*a52*a23*a34
    S410 = a16*a65*a25*a32*a54
    S411 = a61*a12*a25*a32*a54
    S412 = a12*a65*a25*a32*a54
    S413 = a61*a12*a25*a34*a54
    S414 = a16*a65*a25*a54*a34
    S415 = a12*a65*a25*a34*a54
    S4 = S41+S42+S43+S44+S45+S46+S47+S48+S49+S410+S411+S412+S413+S414+S415

    # S5 = sum of S5i
    S51 = a16*a65*a43*a32*a21
    S52 = a61*a45*a34*a23*a12
    S53 = a16*a65*a45*a32*a21
    S54 = a65*a45*a34*a23*a12
    S55 = a16*a65*a45*a34*a21
    S56 = a16*a65*a45*a34*a23
    S57 = a16*a65*a25*a34*a45
    S58 = a16*a65*a25*a32*a43
    S59 = a16*a65*a25*a32*a45
    S510 = a12*a65*a25*a32*a43
    S511 = a12*a65*a25*a34*a45
    S512 = a61*a12*a25*a32*a43
    S513 = a61*a12*a25*a32*a45
    S514 = a61*a12*a25*a34*a45
    S515 = a12*a65*a25*a32*a45
    S5 = S51+S52+S53+S54+S55+S56+S57+S58+S59+S510+S511+S512+S513+S514+S515

    # S6 = sum of S6i
    S61 = a16*a54*a43*a32*a21
    S62 = a56*a45*a34*a23*a12
    S63 = a16*a56*a45*a34*a23
    S64 = a16*a56*a43*a32*a21
    S65 = a16*a56*a45*a32*a21
    S66 = a16*a56*a45*a34*a21
    S67 = a16*a21*a52*a32*a43
    S68 = a16*a21*a52*a34*a45
    S69 = a16*a21*a52*a32*a45
    S610 = a12*a56*a25*a32*a43
    S611 = a12*a56*a25*a32*a45
    S612 = a12*a56*a25*a34*a45
    S613 = a16*a25*a56*a32*a43
    S614 = a16*a56*a25*a34*a45
    S615 = a16*a56*a25*a32*a45
    S6 = S61+S62+S63+S64+S65+S66+S67+S68+S69+S610+S611+S612+S613+S614+S615

    # Sigma is sum of all indices Si
    S = S1+S2+S3+S4+S5+S6

    # MK expression for the A cycle only
    factorA = piAM/S
    return 360*rhoResp*factorA*(10**(6*deltaPh)*eResp/ePsi**2-1.)

# =================================================================

def jPdhBertram(deltaEResp,caM):

    ''' From equations 7 9 and 10 of Bertram's paper
    '''

    kPdhCa  = 0.05 # uM
    kPdhNad = 1    # unitless
    vPdh    = 77   # unitless
    g2      = 0.1  # unitless, volume ratio
    u1      = 1.5  # unitless
    u2      = 1    # unitless

    fbp   = 0.012                    # 15 uM = 0.012 nmol/mg−prot
    kGpdh = 0.024                    # K_GPDH = 5e-4 = 0.024 nmol/min/mg-prot
    jGpdh = kGpdh*np.sqrt(fbp/8.e-4) # 8*10**(-4) nmol/mg-prot = 1 uM

    return 1./(1.+u2*(1.+u1/ (1+caM/kPdhCa)**2)) * vPdh*jGpdh/g2 \
          /(kPdhNad + np.exp(FRT*deltaEResp-lKResp))

# =================================================================

def jHF1F0Mk(deltaGp,deltaPsi):

    ''' Rates of the AKH diagram for the F1F0 complex.
        Derivation is similar to Jresp
    '''

    eG    = np.exp(FRT*deltaGp)
    ePsi  = np.exp(1.5*FRT*deltaPsi)
    ePsiB = ePsi/np.exp(1.5*FRT*deltaPsiB)

    rhoF1 = 0.7 # MK97 value in nmol/mg protein

    a12 = 4.e2
    a21 = 40

    a23star = 5.e3
    atpM    = (0.8*(piM/kF1)*eG*aTot)/(1+(piM/kF1)*eG)
    a23     = a23star*atpM

    a43star  = 5.e4
    adpMFree = aTot/((1/0.8)+(piM/kF1)*eG)
    a43      = a43star*adpMFree*piM*1.e-3

    a32 = 5.e3
    a34 = 1.e2
    a45 = 1.e2
    a54 = 1.e2
    a56 = 1.e3
    a65 = 1.e3
    a25 = 0
    a52 = 0

    a160 = 1.e2
    a16  = a160*ePsiB

    a610 = 4.98e7
    a61  = a610/ePsiB

    piAM = a16*a65*a54*a43*a32*a21

    # S1=sum of S1i
    S11 = a61*a56*a45*a34*a23
    S12 = a65*a54*a43*a32*a21
    S13 = a21*a61*a56*a45*a34
    S14 = a61*a21*a32*a43*a54
    S15 = a56*a61*a43*a32*a21
    S16 = a45*a56*a61*a32*a21
    S17 = a61*a56*a25*a34*a45
    S18 = a61*a56*a25*a32*a45
    S19 = a61*a56*a25*a32*a43
    S110 = a65*a21*a52*a32*a43
    S111 = a21*a65*a52*a32*a45
    S112 = a21*a65*a52*a34*a45
    S113 = a61*a21*a52*a32*a43
    S114 = a61*a21*a52*a32*a45
    S115 = a61*a21*a52*a34*a45
    S1 = S11+S12+S13+S14+S15+S16+S17+S18+S19+S110+S111+S112+S113+S114+S115

    # S2=sum of S2i
    S21 = a34*a45*a56*a61*a12
    S22 = a16*a65*a54*a43*a32
    S23 = a45*a56*a61*a12*a32
    S24 = a12*a65*a54*a43*a32
    S25 = a43*a32*a56*a61*a12
    S26 = a61*a12*a54*a43*a32
    S27 = a61*a12*a52*a43*a32
    S28 = a16*a65*a52*a32*a43
    S29 = a61*a12*a52*a34*a45
    S210 = a12*a65*a52*a34*a45
    S211 = a16*a65*a52*a32*a45
    S212 = a16*a65*a52*a34*a45
    S213 = a12*a65*a52*a32*a43
    S214 = a12*a65*a52*a32*a45
    S215 = a61*a12*a52*a32*a45
    S2 = S21+S22+S23+S24+S25+S26+S27+S28+S29+S210+S211+S212+S213+S214+S215

    # S3=sum of S3i
    S31 = a61*a56*a45*a23*a12
    S32 = a16*a65*a54*a43*a21
    S33 = a16*a65*a54*a43*a23
    S34 = a61*a56*a43*a23*a12
    S35 = a61*a54*a43*a23*a12
    S36 = a65*a54*a43*a23*a12
    S37 = a61*a12*a23*a52*a45
    S38 = a16*a65*a52*a23*a45
    S39 = a12*a65*a52*a23*a45
    S310 = a16*a65*a25*a43*a45
    S311 = a61*a12*a25*a43*a54
    S312 = a12*a65*a25*a43*a54
    S313 = a61*a12*a52*a23*a43
    S314 = a16*a65*a52*a23*a43
    S315 = a12*a65*a52*a23*a43
    S3 = S31+S32+S33+S34+S35+S36+S37+S38+S39+S310+S311+S312+S313+S314+S315

    # S4=sum of S4i
    S41 = a61*a56*a34*a23*a12
    S42 = a16*a65*a54*a32*a21
    S43 = a16*a65*a54*a34*a21
    S44 = a61*a54*a34*a23*a12
    S45 = a16*a65*a54*a34*a23
    S46 = a65*a54*a34*a23*a12
    S47 = a61*a12*a52*a23*a34
    S48 = a12*a65*a52*a23*a34
    S49 = a16*a65*a52*a23*a34
    S410 = a16*a65*a25*a32*a54
    S411 = a61*a12*a25*a32*a54
    S412 = a12*a65*a25*a32*a54
    S413 = a61*a12*a25*a34*a54
    S414 = a16*a65*a25*a54*a34
    S415 = a12*a65*a25*a34*a54
    S4 = S41+S42+S43+S44+S45+S46+S47+S48+S49+S410+S411+S412+S413+S414+S415

    # S5=sum of S5i
    S51 = a16*a65*a43*a32*a21
    S52 = a61*a45*a34*a23*a12
    S53 = a16*a65*a45*a32*a21
    S54 = a65*a45*a34*a23*a12
    S55 = a16*a65*a45*a34*a21
    S56 = a16*a65*a45*a34*a23
    S57 = a16*a65*a25*a34*a45
    S58 = a16*a65*a25*a32*a43
    S59 = a16*a65*a25*a32*a45
    S510 = a12*a65*a25*a32*a43
    S511 = a12*a65*a25*a34*a45
    S512 = a61*a12*a25*a32*a43
    S513 = a61*a12*a25*a32*a45
    S514 = a61*a12*a25*a34*a45
    S515 = a12*a65*a25*a32*a45
    S5 = S51+S52+S53+S54+S55+S56+S57+S58+S59+S510+S511+S512+S513+S514+S515

    # S6=sum of S6i
    S61 = a16*a54*a43*a32*a21
    S62 = a56*a45*a34*a23*a12
    S63 = a16*a56*a45*a34*a23
    S64 = a16*a56*a43*a32*a21
    S65 = a16*a56*a45*a32*a21
    S66 = a16*a56*a45*a34*a21
    S67 = a16*a21*a52*a32*a43
    S68 = a16*a21*a52*a34*a45
    S69 = a16*a21*a52*a32*a45
    S610 = a12*a56*a25*a32*a43
    S611 = a12*a56*a25*a32*a45
    S612 = a12*a56*a25*a34*a45
    S613 = a16*a25*a56*a32*a43
    S614 = a16*a56*a25*a34*a45
    S615 = a16*a56*a25*a32*a45
    S6 = S61+S62+S63+S64+S65+S66+S67+S68+S69+S610+S611+S612+S613+S614+S615

    # Sigma is sum of all Si
    S = S1+S2+S3+S4+S5+S6
    factorA = piAM/S

    return -180*rhoF1*factorA*(10**(3*deltaPh)*eG/ePsi**2-1)

# =================================================================

def jHAntMk(deltaGpM,deltaGpC,deltaPsi):

    f       = 0.5
    jAntMax = 1000   # unité nmol/min/mg protein

    ePsi  = np.exp(-FRT*deltaPsi)
    eGM   = np.exp( FRT*deltaGpM)
    eGC   = np.exp( FRT*deltaGpC)

    return jAntMax*(1.- piC/piM * eGC/eGM *ePsi)/ ( (1+piC/kF1*eGC *ePsi**f)*(1+kF1/piM/eGM))

# =================================================================

def jHUniMk(caC,deltaP):

    jMax     = 300     # nmol/min/mg protein
    kTrans   = 0.0152  # nmol/mg prot = 19 \mu M
    kAct     = 3.04e-4 # nmol/mg prot = 0.38 \mu M 
    dPsiStar = 91      # mV
    L        = 110
    na       = 2.8

    A = caC/kTrans
    B = caC/kAct
    C = deltaP - dPsiStar
    return jMax*((A*(1+A)**3)/((1+A)**4+L/(1+B)**na))*((2*FRT*C)/(1-np.exp(-2*FRT*C)))


