function [cal_age,prob,hp_68p2,hp_95p4] = epdf(X,A,B,delta)
%% function for estimating empirical pdf and highest posterior probability of 
%% a modeled calendar age
%INPUT
%X: vector containing random numbers obtained from the MCMC run 
%A: early bound of the age sequence
%B: late bound of the age sequence
%delta: resolution in years
%year_scale: scale of year(BC/AD or BP)
%
%OUTPUT
%cal_age: calendar ages 
%prob: probability of the calendar ages
%hp_68p2: 68.2% highest posterior probability
%hp_95p4: 95.4% highest posterior probability
%% building up the empirical pdf
%generate calendar age points at which pdf will be estimated
cal_age = A:B;
%resample at delta-yr interval
cal_age = cal_age(mod(cal_age,delta) == 0);
cal_age = cal_age';
M = length(cal_age);
N = length(X);
prob = zeros(M,1);
hp_68p2 = zeros(M,1);
hp_95p4 = zeros(M,1);
F = zeros(M,1);
h = 0.000000001;        % Small value closer to zero for evaluating
                        % numerical differentiation.
% estimating CDF by its definition
for i = 1:M
    p = 0;              % True Probability
    q = 0;              % False Probability
    for j = 1:N
        if X(j)<=cal_age(i)   % Definition of CDF
            p = p + 1;
        else
            q = q + 1;
        end
    end
    F(i) = p/(p + q);   % Calulating Probability
end
% estimating PDF by differentiation of CDF
for k = 1:M
    fxph = interp1(cal_age,F,cal_age(k) + h,'spline');  % Interpolating value of F(x+h)
    fxmh = interp1(cal_age,F,cal_age(k) - h,'spline');  % Interpolating value of F(x-h)
    prob(k) = (fxph - fxmh)/(2*h); 
    if prob(k) < 0
       prob(k) = 0;
    end        
end                                         
prob = smooth(prob);  
% normalizing to 1
prob = prob./sum(prob);
%% calculate highest posterior probabilities
calprob = [cal_age prob];
hpd = calprob(:,1:2);
hpd = sortrows(hpd, 2);
hpd(:,3) = cumsum(hpd(:,2));
%one-sigma highest posterior probability
hpd68_2 = hpd(hpd(:,3) >= 1-erf(1/sqrt(2)), :);
hpd68_2 = sortrows(hpd68_2,1);
id0 = ismember(cal_age,hpd68_2(:,1)); 
hp_68p2(id0) = hpd68_2(:,2);
%two-sigma highest posterior probability
hpd95_4 = hpd(hpd(:,3) >= 1-erf(2/sqrt(2)), :);
hpd95_4 = sortrows(hpd95_4,1);
id1 = ismember(cal_age,hpd95_4(:,1)); 
hp_95p4(id1) = hpd95_4(:,2);
return;