function [epdfs,hpd68_2,hpd95_4] = mcmc2pdf(mcmcsamples,A,B,delta,year_scale,R)
%% function for calculating the pdf of calendar ages and save results to a file 
%INPUT
%mcmcsamples: randomly generated calendar ages of the raddiocaron ages and boundaries
%A: early boundary of the age sequence
%B: late boundary of the age sequence
%delta: resolution of the age points in years
%year_scale: scale of year(BC/AD or BP)
%OUTPUT
%prob: vector containing estimated PDF at the age points
%hp_68p2: 68.2% highest probability 
%hp_95p4: 95.4% highest probability
%% preallocate size
if strcmpi(year_scale,'BC/AD') == 1
   x = A:B;
else
   x = B:A;
end
%resample at delta-yr intervals
x = x(mod(x,delta) == 0);
M = length(x);
N = size(mcmcsamples,2);
epdfs = zeros(M,N+1);
hpd68_2 = zeros(M,N+1);
hpd95_4 = zeros(M,N+1);
%% get results
for i = 1:N
    [cal_age,prob,hp_68p2,hp_95p4] = epdf(mcmcsamples(:,i),A,B,delta,year_scale);
    epdfs(:,i+1) = prob;
    hpd68_2(:,i+1) = hp_68p2;
    hpd95_4(:,i+1) = hp_95p4;
end
epdfs(:,1) = cal_age;
hpd68_2(:,1) = cal_age; 
hpd95_4(:,1) = cal_age;
%% save results to a file
outfiles = {'pdfs.txt','hp68_2.txt','hp95_4.txt'};
results = [epdfs hpd68_2 hpd95_4];
for i = 1:length(outfiles)
    savepdf(outfiles{i},R,year_scale,results(:,(N+1)*i-N:(N+1)*i));
end
return;
%%
function [cal_age,prob,hp_68p2,hp_95p4] = epdf(X,A,B,delta,year_scale)
%% function for estimating the empirical pdf and highest probability from MCMC runs
%INPUT
%X: vector specifying random variables obtained from MCMC run 
%A: early bound of the age sequence
%B: late bound of the age sequence
%delta: resolution in years
%year_scale: scale of year(BC/AD or BP)
%
%OUTPUT
%cal_age: calendar age points at which PDF was estimated 
%prob: vector specifying estimated PDF at age points
%hp_68p2: 68.2% highest posterior probability
%hp_95p4: 95.4% highest posterior probability
%% building up an epdf
%generate points at which pdf will be estimated
if strcmpi(year_scale,'BC/AD') == 1
   cal_age = A:B;
else
   cal_age = B:A;
end
%resample at delta-yr interval
cal_age = cal_age(mod(cal_age,delta) == 0);
cal_age = cal_age';
M = length(cal_age);
N = length(X);
prob = zeros(M,1);
hp_68p2 = zeros(M,1);
hp_95p4 = zeros(M,1);
F = zeros(M,1);
h = 0.000000001;        % Small value closer to zero for evaluating
                        % numerical differentiation.
% Estimating CDF by its definition
for i = 1:M
    p = 0;              % True Probability
    q = 0;              % False Probability
    for j = 1:N
        if X(j)<=cal_age(i)   % Definition of CDF
            p = p + 1;
        else
            q = q + 1;
        end
    end
    F(i) = p/(p + q);   % Calulating Probability
end
% Estimating PDF by differentiation of CDF
for k = 1:M
    fxph = interp1(cal_age,F,cal_age(k) + h,'spline');  % Interpolating value of F(x+h)
    fxmh = interp1(cal_age,F,cal_age(k) - h,'spline');  % Interpolating value of F(x-h)
    prob(k) = (fxph - fxmh)/(2*h); 
    if prob(k) < 0
       prob(k) = 0;
    end        
end                                         
prob = smooth(prob);  
% Normalizing to 1
prob = prob./sum(prob);
%% calculate higher probabilities
calprob = [cal_age prob];
hpd = calprob(:,1:2);
hpd = sortrows(hpd, 2);
hpd(:,3) = cumsum(hpd(:,2));
%1 sigma prob
hpd68_2 = hpd(hpd(:,3) >= 1-erf(1/sqrt(2)), :);
hpd68_2 = sortrows(hpd68_2,1);
id0 = ismember(cal_age,hpd68_2(:,1)); 
hp_68p2(id0) = hpd68_2(:,2);
%2 sigma prob
hpd95_4 = hpd(hpd(:,3) >= 1-erf(2/sqrt(2)), :);
hpd95_4 = sortrows(hpd95_4,1);
id1 = ismember(cal_age,hpd95_4(:,1)); 
hp_95p4(id1) = hpd95_4(:,2);
return;
%
function [] = savepdf(filename,R,year_scale,pdfs)
%% function for saving the empirical pdfs to a file
%INPUT
%filename: file name to which for data to be saved
%R: information of radiocarbon ages assembled in a structure
%year_scale: scale of year (BC/AD or BP)
%pdfs: probabilities of the calendar ages
%% creat a header
M = length(R);      %number of phases
N = zeros(1,M);     %number of ages in each phase
for i = 1:M
    N(i) = length(R(i).age); 
end
header = cell(1,2*M+sum(N)+1); 
ind_alpha = cumsum([1 N(1:end-1)+2]); %index of alpha in the age sequence
ind_b_theta = cumsum([1 N(1:end-1)+2]) + 1; %index of beginning of theta in the sequence 
%ind_e_theta = cumsum([1 N(1:end-1)+2]) + N; %index of end of theta in the sequence
ind_beta = cumsum([1 N(1:end-1)+2]) + N + 1;  %index of beta in the age sequence
header{1} = strcat('Age (',year_scale,')');
for i = 1:M
    %header{1,1+ind_alpha(i)} = strcat('Alpha_',num2str(i)); %alphas
    header{1,1+ind_alpha(i)} = strcat('Phase_',num2str(i),' early boundary');
    %header{1,1+ind_beta(i)} = strcat('Beta_',num2str(i)); %betas
    header{1,1+ind_beta(i)} = strcat('Phase_',num2str(i),' late boundary');
    for j = 1:N(i)
        %header{1,1+ind_b_theta(i)+j-1}=strcat('Theta_',[num2str(i),',',num2str(j)]); %thetas
        header{1,1+ind_b_theta(i)+j-1} = R(i).lab_code{j};
    end
end
%% write header and data to a file
fmt = repmat('%25g ',1,2*M+sum(N)); 
fmt = ['%25d ' fmt];
fid = fopen(filename,'wt');
fprintf(fid,'%25s ',header{1:end});
fprintf(fid,'\n');
for k = 1:size(pdfs,1)
    fprintf(fid,fmt,pdfs(k,:));
    fprintf(fid,'\n');
end    
fclose(fid);
return;