function [theta,alpha,beta] = initialization(P,A,B,year_scale,cal_dataset,phase_relationship)
%% function for initializing model parameters
%INPUT
%P: structure containing laboratory data of the phases of a sequence
%A: early boundary of the age sequence
%B: late boundary of the age sequence
%year_scale: scale of year ("BC/AD" or "BP")
%cal_dataset: extracted calibration curve in the F14C space
%phase_relationship: vector containing the relationship of two phases ('overlapping',disjoint','contiguous')
%OUTPUT
%theta: structure containing the calendar ages of the phases
%alpha: vector containing the calendar ages of the early boundaries of the phases
%beta: vector containing the calendar ages of the late boundaries of the phases
%% obtain calibrated ages of the radiocarbon ages in the phases
M = length(P); % number of phases
alpha = zeros(1,M);
beta = zeros(1,M);
theta = struct('age',[]);
for i = 1:M
    x = calib(P(i).f14C,P(i).ferr,year_scale,cal_dataset);  
    if i == 1 % the first phase 
        if strcmpi(year_scale,'BC/AD') == 1 
            true = all(x > A) && all(diff(x)~=0); % obtain unique ages 
            while(~true)
                 x = calib(P(i).f14C,P(i).ferr,year_scale,cal_dataset);  
                 true = all(x > A) && all(diff(x)~=0);
            end
            if strcmpi(P(i).ordering,'unordered') == 1
                theta(i).age = x; % no ordering               
            elseif strcmpi(P(i).ordering,'coeval') == 1
                theta(i).age = repmat(round(mean(x)),1,length(x)); % make all(diff(theta.age))=0    
            elseif strcmpi(P(i).ordering,'ordered') == 1
                theta(i).age = sort(x,'ascend'); % make all(diff(theta.age))>0
            end    
        elseif strcmpi(year_scale,'BP') == 1
            true = all(x < A) && all(diff(x)~=0);
            while(~true)
                x = calib(P(i).f14C,P(i).ferr,year_scale,cal_dataset);  
                true = all(x < A) && all(diff(x)~=0);
            end
            if strcmpi(P(i).ordering,'unordered') == 1
                theta(i).age = x; % no ordering                 
            elseif strcmpi(P(i).ordering,'coeval') == 1
                theta(i).age = repmat(round(mean(x)),1,length(x)); % make all(diff(theta.age))=0     
            elseif strcmpi(P(i).ordering,'ordered') == 1
                theta(i).age = sort(x,'descend'); % make all(diff(theta.age))<0
            end
        end
    elseif i > 1 && i < M %internal phases    
        if strcmpi(year_scale,'BC/AD') == 1
            if strcmpi(phase_relationship(i-1),'overlapping') == 1
                true = min(theta(i-1).age) < min(x) && min(x) < max(theta(i-1).age) && max(theta(i-1).age) < max(x) && all(diff(x)~=0);
            else %contiguous and disjoint cases    
                true = max(theta(i-1).age) < min(x) && all(diff(x)~=0);
            end    
            while(~true)
                x = calib(P(i).f14C,P(i).ferr,year_scale,cal_dataset);  
                if strcmpi(phase_relationship(i-1),'overlapping') == 1
                    true = min(theta(i-1).age) < min(x) && min(x) < max(theta(i-1).age) && max(theta(i-1).age) < max(x) && all(diff(x)~=0);
                else %contiguous and disjoint cases    
                    true = max(theta(i-1).age) < min(x) && all(diff(x)~=0);
                end    
            end 
            if strcmpi(P(i).ordering,'unordered') == 1
                theta(i).age = x; % no ordering                 
            elseif strcmpi(P(i).ordering,'coeval') == 1
                theta(i).age = repmat(round(mean(x)),1,length(x)); % make all(diff(theta.age))=0
            elseif strcmpi(P(i).ordering,'ordered') == 1  
                theta(i).age = sort(x,'ascend'); % make all(diff(theta.age))>0
            end    
        elseif strcmpi(year_scale,'BP') == 1
            if strcmpi(phase_relationship(i-1),'overlapping') == 1
                true = max(theta(i-1).age) > max(x) && max(x) > min(theta(i-1).age) && min(theta(i-1).age) > min(x) && all(diff(x)~=0);
            else % dcontiguous and disjoint cases
                true = min(theta(i-1).age) > max(x) && all(diff(x)~=0);
            end
            while(~true)
                x = calib(P(i).f14C,P(i).ferr,year_scale,cal_dataset);  
                if strcmpi(phase_relationship(i-1),'overlapping') == 1
                    true = max(theta(i-1).age) > max(x) && max(x) > min(theta(i-1).age) && min(theta(i-1).age) > min(x) && all(diff(x)~=0);
                else % dcontiguous and disjoint cases
                    true = min(theta(i-1).age) > max(x) && all(diff(x)~=0);
                end
            end 
            if strcmpi(P(i).ordering,'unordered') == 1
                theta(i).age = x; % no ordering                 
            elseif strcmpi(P(i).ordering,'coeval') == 1
                theta(i).age = repmat(round(mean(x)),1,length(x)); % make all(diff(theta.age))=0
            elseif strcmpi(P(i).ordering,'ordered') == 1
                theta(i).age = sort(x,'descend'); % make all(diff(theta.age))<0
            end    
        end 
    else % the last phase
        if strcmpi(year_scale,'BC/AD') == 1
            if strcmpi(phase_relationship(i-1),'overlapping') == 1
                true = min(theta(i-1).age) < min(x) && min(x) < max(theta(i-1).age) && max(theta(i-1).age) < max(x) && max(x) < B && all(diff(x)~=0);
            else %contiguous and disjoint cases    
                true = max(theta(i-1).age) < min(x) && max(x) < B && all(diff(x)~=0);
            end    
            while(~true)
                x = calib(P(i).f14C,P(i).ferr,year_scale,cal_dataset);  
                if strcmpi(phase_relationship(i-1),'overlapping') == 1
                    true = min(theta(i-1).age) < min(x) && min(x) < max(theta(i-1).age) && max(theta(i-1).age) < max(x) && max(x) < B && all(diff(x)~=0);
                else %contiguous and disjoint cases    
                    true = max(theta(i-1).age) < min(x) && max(x) < B && all(diff(x)~=0);
                end    
            end
            if strcmpi(P(i).ordering,'unordered') == 1
                theta(i).age = x; % no ordering                 
            elseif strcmpi(P(i).ordering,'coeval') == 1
                theta(i).age = repmat(round(mean(x)),1,length(x)); % make all(diff(theta.age))=0
            elseif strcmpi(P(i).ordering,'ordered') == 1
                theta(i).age = sort(x,'ascend'); % make all(diff(theta.age))>0
            end    
        elseif strcmpi(year_scale,'BP') == 1
            if strcmpi(phase_relationship(i-1),'overlapping') == 1
                true = max(theta(i-1).age) > max(x) && max(x) > min(theta(i-1).age) && min(theta(i-1).age) > min(x) && min(x) > B && all(diff(x)~=0);
            else % dcontiguous and disjoint cases
                true = min(theta(i-1).age) > max(x) && min(x) > B && all(diff(x)~=0);    
            end
            while(~true)
                x = calib(P(i).f14C,P(i).ferr,year_scale,cal_dataset);  
                if strcmpi(phase_relationship(i-1),'overlapping') == 1
                    true = max(theta(i-1).age) > max(x) && max(x) > min(theta(i-1).age) && min(theta(i-1).age) > min(x) && min(x) > B && all(diff(x)~=0);
                else % dcontiguous and disjoint cases
                    true = min(theta(i-1).age) > max(x) && min(x) > B && all(diff(x)~=0);    
                end
            end
            if strcmpi(P(i).ordering,'unordered') == 1
                theta(i).age = x; % no ordering                 
            elseif strcmpi(P(i).ordering,'coeval') == 1
                theta(i).age = repmat(round(mean(x)),1,length(x)); % make all(diff(theta.age))=0
            elseif strcmpi(P(i).ordering,'ordered') == 1
                theta(i).age = sort(x,'descend'); % make all(diff(theta.age))<0
            end    
        end 
    end
end
%% obtain the early boundary of the phases
for i = 1:M
    if i == 1 % the first phase
        if strcmpi(year_scale,'BC/AD') == 1
            x = (A+min(theta(i).age))/2;
            alpha(i) = trirnd(x,min(theta(i).age),min(theta(i).age),1);
        elseif strcmpi(year_scale,'BP') == 1
            x = (A+max(theta(i).age))/2;
            alpha(i) = trirnd(max(theta(i).age),max(theta(i).age),x,1);
        end
    else %other phases
        if strcmpi(year_scale,'BC/AD') == 1
            if strcmpi(phase_relationship(i-1),'overlapping') == 1
                x = (min(theta(i-1).age)+min(theta(i).age))/2;
                alpha(i) = trirnd(x,min(theta(i).age),min(theta(i).age),1);
            elseif strcmpi(phase_relationship(i-1),'contiguous') == 1
                x = (max(theta(i-1).age)+min(theta(i).age))/2;
                alpha(i) = trirnd(max(theta(i-1).age),x,min(theta(i).age),1);
            elseif strcmpi(phase_relationship(i-1),'disjoint') == 1
                x = (max(theta(i-1).age)+min(theta(i).age))/2;
                alpha(i) = trirnd(x,min(theta(i).age),min(theta(i).age),1);
            end
        elseif strcmpi(year_scale,'BP') == 1
            if strcmpi(phase_relationship(i-1),'overlapping') == 1
                x = (max(theta(i-1).age)+max(theta(i).age))/2;
                alpha(i) = trirnd(max(theta(i).age),max(theta(i).age),x,1);
            elseif strcmpi(phase_relationship(i-1),'contiguous') == 1
                x = (min(theta(i-1).age) + max(theta(i).age))/2;
                alpha(i) = trirnd(max(theta(i).age),x,min(theta(i-1).age),1);
            elseif strcmpi(phase_relationship(i-1),'disjoint') == 1
                x = (min(theta(i-1).age) + max(theta(i).age))/2;
                alpha(i) = trirnd(max(theta(i).age),max(theta(i).age),x,1);
            end
        end 
    end
    alpha(i) = round(alpha(i));
end   
%% obtain the late boundary of the phases
for i = 1:M
    if i < M % first M-1 phases
        if strcmpi(year_scale,'BC/AD') == 1
            if strcmpi(phase_relationship(i),'overlapping') == 1
                x = (max(theta(i).age)+max(theta(i+1).age))/2;
                beta(i) = trirnd(max(theta(i).age),max(theta(i).age),x,1);
            elseif strcmpi(phase_relationship(i),'contiguous') == 1
                beta(i) = alpha(i+1);
            elseif strcmpi(phase_relationship(i),'disjoint') == 1
                x = (max(theta(i).age)+min(theta(i+1).age))/2;
                beta(i) = trirnd(max(theta(i).age),max(theta(i).age),x,1);
            end
        elseif strcmpi(year_scale,'BP') == 1
            if strcmpi(phase_relationship(i),'overlapping') == 1
                x = (min(theta(i).age)+min(theta(i+1).age))/2;
                beta(i) = trirnd(x,min(theta(i).age),min(theta(i).age),1);
            elseif strcmpi(phase_relationship(i),'contiguous') == 1
                beta(i) = alpha(i+1);
            elseif strcmpi(phase_relationship(i),'disjoint') == 1
                x = (min(theta(i).age)+max(theta(i+1).age))/2;
                beta(i) = trirnd(x,min(theta(i).age),min(theta(i).age),1);
            end
        end 
    else % the last phase
        if strcmpi(year_scale,'BC/AD') == 1
            x = (B+max(theta(i).age))/2;
            beta(i) = trirnd(max(theta(i).age),max(theta(i).age),x,1);
        elseif strcmpi(year_scale,'BP') == 1
            x = (B+min(theta(i).age))/2;
            beta(i) = trirnd(x,min(theta(i).age),min(theta(i).age),1);
        end
    end
    beta(i) = round(beta(i));
end  
return;
%%
function cal_age = calib(F_14C_val,F_14C_err,year_scale,cal_dataset)
%% Function for 14C age calibration using Bayesian highest posterior density analysis
%INPUT
% F_14C: laboratory 14C age in F14C space
% F_err: laboratory 14C age error in F14C space
% year_scale: scale of year to be reported (e.g. BC/AD or BP)
% cal_dataset: extracted calibration curve in the F14C space 
%OUTPUT
% cal_age: calibrated age in year_scale
%%
N = length(F_14C_val);
cal_age = zeros(1,N);
curve_cal_age = cal_dataset(:,1);
curve_F14_val = cal_dataset(:,2);
curve_F14_err = cal_dataset(:,3);
%% interpolate the calendar age in the curve to annual resolution
interpres = 1;
hi_curve_cal_age = curve_cal_age(1):interpres:curve_cal_age(end);
hi_curve_F14_val = interp1(curve_cal_age, curve_F14_val, hi_curve_cal_age);
hi_curve_F14_err = interp1(curve_cal_age, curve_F14_err, hi_curve_cal_age);
%% Calculate probability for every calendar year in the F14 space
cal_prob = zeros(length(hi_curve_cal_age),2);
cal_prob(:,1) = hi_curve_cal_age;
for i = 1:N
    a = (F_14C_val(i) - hi_curve_F14_val).^2;
    b = 2*(F_14C_err(i).^2 + hi_curve_F14_err.^2);
    c = sqrt(F_14C_err(i).^2 + hi_curve_F14_err.^2);
    cal_prob(:,2) = exp(-a./b)./c;
    % normalize the probabilities to 1
    cal_prob(:,2) = cal_prob(:,2)/sum(cal_prob(:,2));
    cal_age(i) = durand(cal_prob(:,1),cal_prob(:,2),1,1);
end
%% convert to BC/AD scale if needed
if strcmpi(year_scale,'BC/AD') == 1
   cal_age = 1950 - cal_age;
end
return;
%%
function x = durand(values,prob,M,N)
%% Function for drawing a random number from a list at given probability
values = values(:);
prob = prob(:);
if sum(prob)~=1
   prob = prob/sum(prob);
end
L = length(prob);
K = M*N;
psup = cumsum(prob);
pinf = [0; psup(1:end-1)];
Pinf = kron(ones(1,K),pinf(:));
Psup = kron(ones(1,K),psup(:));
u = rand(1,K);
U = kron(ones(L,1),u);
C = (U>Pinf) & (U<Psup);
V = kron(values(:),ones(1,K));
X = V.*C;
x = sum(X);
x = reshape(x,M,N);
return;