%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                     Bayesian radiocarbon age modeling                   % 
%                     MatCalib v2.0                                       % 
%                     Copyright: Shiyong Yu                               %                                  
%                     E-mail: shiyong.yu@gmail.com                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clc;
clear;
%% input the information of radiocarbon ages to be modeled
P(1).lab_code = {'AMS-01','AMS-02','AMS-03','AMS-04'};  % laboratory number
P(1).age = [5880,5760,5690,5510];              % laboratory 14C age
P(1).err = [25,20,30,20];                % laboratory 14C age error
P(1).rage = [0,0,0,0];                % reservoir age
P(1).rerr = [0,0,0,0];                % reservoir age error 
P(1).ordering = 'ordered';    % ordering constraint (unordered|coeval|ordered)
P(2).lab_code = {'AMS-05','AMS-06','AMS-07','AMS-08','AMS-09'}; 
P(2).age = [5490,5420,5350,5260,5120];   
P(2).err = [20,20,20,20,30];
P(2).rage = [0,0,0,0,0];       
P(2).rerr = [0,0,0,0,0];
P(2).ordering = 'ordered';
P(3).lab_code = {'AMS-10','AMS-11','AMS-12'}; 
P(3).age = [4920,4890,4900];   
P(3).err = [20,20,20];
P(3).rage = [0,0,0];       
P(3).rerr = [0,0,0];
P(3).ordering = 'unordered';
% specify the relationship of every two neighboring phases (overlapping|contiguous|disjoint)
phase_relationship = {'overlapping','disjoint'};  % phase relationship  
%% set up model parameters
cal_curve = 'intCal20';    % specify the file name of calibration curve
delta = 5;                 % round ages to the nearest 5 years
year_scale = 'BP';         % scale of year to be reported (BC/AD or BP)
nsamples = 1000;           % number of random samples to be saved
burn_in = 1000;            % steps of MCMC run when samples will be saved  
thin = 20;                 % steps in every which samples will be saved 
%% convert 14C ages to F14C space
M = length(P);
for i = 1:M
    [P(i).f14C,P(i).ferr] = radiocarbon2f(P(i).age,P(i).err,P(i).rage,P(i).rerr);
end
%% estimate the early and late boundary of the sequence
[A,B] = sequence_bound(P,year_scale,cal_curve);
%% read the calibration curve and convert to F14C space
cal_dataset = read_curve(cal_curve,A,B,year_scale);                     
%% initialize model parameters
[theta,alpha,beta] = initialization(P,A,B,year_scale,cal_dataset,phase_relationship);
%% infer model parameters and save the results 
% disp('Begin Bayesian age modeling...');
mcmcsamples = mcmc(theta,alpha,beta,P,A,B,year_scale,cal_dataset,phase_relationship,nsamples,thin,burn_in);
% disp('Bayesian age modeling completed');
%% estimate empirical pdf and highest posterior density regions of the parameters and save results
[pdfs,hpd68_2,hpd95_4] = mcmc2pdf(mcmcsamples,A,B,delta,year_scale,P);
%% calculate 95.4% and 68.2% highest density regions and median probability age of the parameters
[ALPHA,BETA,THETA,CAGES] = pdf2hpd(pdfs,delta,P,year_scale); %results were saved in three structure arrays
Table = struct2table(CAGES);
%% plot modeled ages
figure(1)
plot_ages(pdfs,CAGES,year_scale);
%% calculate the differnce between two modeled ages
x1 = mcmcsamples(:,1);
x2 = mcmcsamples(:,6);
bin_size = 5;
[diff_pdfs,AGE_DIFF] = age_difference(x1,x2,bin_size);
%% plot age difference
figure(2)
plot_difference(diff_pdfs);
%% clean up the work space
clear P A B alpha beta theta cal_dataset cal_curve i M burn_in nsamples ...
    thin mcmcsamples phase_relationship year_scale delta x1 x2 bin_size;