%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                     Bayesian radiocarbon age modeling                   %
%                     MatCalib v2.0                                       % 
%                     Copyright: Shiyong Yu                               %                                  
%                     E-mail: shiyong.yu@gmail.com                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clc;
clear;
%% input the information of radiocarbon ages of the lacustrine facies
P(1).lab_code = {'LuA-5456','LuA-5455','LuA-5454'};  % laboratory number
P(1).age = [9890,8220,7860];              % laboratory 14C age
P(1).err = [100,110,95];                % laboratory 14C age error
P(1).rage = [0,0,0];                % reservoir age
P(1).rerr = [0,0,0];                % reservoir age error 
P(1).ordering = 'ordered';    % ordering constraint (unordered|coeval|ordered)
P(1).depth = [-783.0,-674.5,-601.5]; % sampling depths
%% input the information of radiocarbon ages of the lagoonal facies
P(2).lab_code = {'LuA-5453','LuA-5452','LuA-5445','LuA-5451','LuA-5450',...
    'LuA-5449','LuA-5448'}; 
P(2).age = [6415,5455,4875,4800,4850,4575,2935];   
P(2).err = [90,90,90,95,90,95,80];
P(2).rage = [108,108,108,108,108,108,0];       
P(2).rerr = [24,24,24,24,24,24,0];
P(2).ordering = 'ordered';
P(2).depth = [-567.5,-508,-436,-376,-274.5,-180,-89.5];
%% specifying depth unit
depth_unit = 'cm'; 
%% specify the relationship of every two neighboring phases (overlapping|contiguous|disjoint)
phase_relationship = {'disjoint'};  % phase relationship  
%% specifying depths of phase boundaries
alpha_depth = [-850,-575];
beta_depth = [-575,0];
%% set up model parameters
cal_curve = 'intCal20';    % specify the file name of calibration curve
delta = 5;                 % round ages to the nearest 5 years
year_scale = 'BP';         % scale of year to be reported (BC/AD or BP)
nsamples = 1000;           % number of random samples to be saved
burn_in = 1000;            % steps of MCMC run when samples will be saved  
thin = 20;                 % steps in every which samples will be saved 
%% convert 14C ages to F14C space
M = length(P);
for i = 1:M
    [P(i).f14C,P(i).ferr] = radiocarbon2f(P(i).age,P(i).err,P(i).rage,P(i).rerr);
end
%% estimate the early and late boundary of the sequence
[A,B] = sequence_bound(P,year_scale,cal_curve);
%% read the calibration curve and convert to F14C space
cal_dataset = read_curve(cal_curve,A,B,year_scale);                     
%% initialize model parameters
[theta,alpha,beta] = initialization(P,A,B,year_scale,cal_dataset,phase_relationship);
%% infer model parameters and save the results 
% disp('Begin Bayesian age modeling...');
mcmcsamples = mcmc(theta,alpha,beta,P,A,B,year_scale,cal_dataset,phase_relationship,nsamples,thin,burn_in);
% disp('Bayesian age modeling completed');
%% estimate empirical pdf and highest posterior density regions of the parameters and save results
[pdfs,hpd68_2,hpd95_4] = mcmc2pdf(mcmcsamples,A,B,delta,year_scale,P);
%% calculate 95.4% and 68.2% highest density regions and median probability age of the parameters
[ALPHA,BETA,THETA,CAGES] = pdf2hpd(pdfs,delta,P,year_scale); %results were saved in three structure arrays
%Table = struct2table(CAGES)
%% plot results
figure(1)
plot_age_depth(pdfs,CAGES,P,alpha_depth,beta_depth,depth_unit,year_scale);
%% calculate the length of the sedimentary hiatus and save the results
x1 = mcmcsamples(:,5);
x2 = mcmcsamples(:,6);
bin_size = 10;
[diff_pdfs,AGE_DIFF] = age_difference(x1,x2,bin_size);
%% plot the pdf of the hiatus
figure(2)
plot_difference(diff_pdfs);
%% clean up the work space
clear P A B alpha beta theta cal_dataset cal_curve i M burn_in nsamples ...
    thin mcmcsamples phase_relationship year_scale delta x1 x2 bin_size;