%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                     Bayesian radiocarbon age modeling                   %
%                     MatCalib v2.0                                       % 
%                     Copyright: Shiyong Yu                               %                                  
%                     E-mail: shiyong.yu@gmail.com                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clc;
clear;
%% input the radiocarbon ages of the Longshan cultural sites
P(1).lab_code = {'ZK-1303B','ZK-1303A','ZK-2306','BK82035','BK82034',...
'BK82038','BK82036','BK82041','BK82039'};  % laboratory number
P(1).age = [4090,4020,3960,3920,3880,3870,3800,3740,3570]; % laboratory 14C age
P(1).err = [80,80,105,70,70,70,90,100,80];                % laboratory 14C age error
P(1).rage = [0,0,0,0,0,0,0,0,0];                % reservoir age
P(1).rerr = [0,0,0,0,0,0,0,0,0];                % reservoir age error
P(1).ordering = 'unordered';    % ordering constraint (unordered|coeval|ordered)
%% input the radiocarbon ages of the Yueshi cultural sites
P(2).lab_code = {'ZK-2311','ZK-2309','ZK-2310',...
'ZK-0868','ZK-0871','ZK-0869','ZK-0872','ZK-0870','BK82027','BK80036'}; 
P(2).age = [3760,3760,3590,3450,3430,3420,3340,3330,3270,3270];   
P(2).err = [145,70,130,90,80,80,80,80,100,70];
P(2).rage = [0,0,0,0,0,0,0,0,0,0];       
P(2).rerr = [0,0,0,0,0,0,0,0,0,0];
P(2).ordering = 'unordered';
%% specify the relationship of every two neighboring phases (overlapping|contiguous|disjoint)
phase_relationship = {'overlapping'};  % phase relationship
%% set up model parameters
cal_curve = 'intCal20';    % specify the file name of calibration curve
delta = 5;                 % round ages to the nearest 5 years
year_scale = 'BC/AD';         % scale of year to be reported (BC/AD or BP)
nsamples = 1000;           % number of random samples to be saved
burn_in = 1000;            % steps of MCMC run when samples will be saved  
thin = 20;                 % steps in every which samples will be saved 
%% convert 14C ages to F14C space
M = length(P);
for i = 1:M
    [P(i).f14C,P(i).ferr] = radiocarbon2f(P(i).age,P(i).err,P(i).rage,P(i).rerr);
end
%% estimate the early and late boundary of the sequence
[A,B] = sequence_bound(P,year_scale,cal_curve);
%% read the calibration curve and convert to F14C space
cal_dataset = read_curve(cal_curve,A,B,year_scale);                     
%% initialize model parameters
[theta,alpha,beta] = initialization(P,A,B,year_scale,cal_dataset,phase_relationship);
%% infer model parameters and save the results 
% disp('Begin Bayesian age modeling...');
mcmcsamples = mcmc(theta,alpha,beta,P,A,B,year_scale,cal_dataset,phase_relationship,nsamples,thin,burn_in);
% disp('Bayesian age modeling completed');
%% estimate empirical pdf and highest posterior density regions of the parameters and save results
[pdfs,hpd68_2,hpd95_4] = mcmc2pdf(mcmcsamples,A,B,delta,year_scale,P);
%% calculate 95.4% and 68.2% highest density regions and median probability age of the parameters
[ALPHA,BETA,THETA,CAGES] = pdf2hpd(pdfs,delta,P,year_scale); %results were saved in three structure arrays
%Table = struct2table(CAGES)
%% plot results
figure(1)
plot_ages(pdfs,CAGES,year_scale);
%% calculate the length of coexistence of the two cultures
x1 = mcmcsamples(:,11);
x2 = mcmcsamples(:,12);
bin_size = 10;
[diff_pdfs,AGE_DIFF] = age_difference(x1,x2,bin_size);
%% plot the pdf of the length of coexistence
figure(2)
plot_difference(diff_pdfs);
%% clean up the work space
clear P A B alpha beta theta cal_dataset cal_curve i M burn_in nsamples ...
    thin mcmcsamples phase_relationship year_scale delta x1 x2 bin_size;